<?php
use Illuminate\Database\Capsule\Manager as Capsule;

/**
 * Client Monthly Retention Report
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan D. Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 *
 *************************************************
 *   1.0 Initial Report Released.  - Juan D. Bolivar
 ************************************************
 */

/**
 * Client_Monthly_Retention Report Class
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan D. Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 */
class Client_Monthly_Retention extends Report
{
    private $lang;

    protected $featureSet = 'accounts';

    function __construct($user = null, $customer = null)
    {
        $this->lang = lang('Client Monthly Retention From Last Year');
        parent::__construct($user, $customer);
    }

    /**
     * Report Process Method
     *
     * @return null - direct output
     */
    function process()
    {
        $this->SetDescription($this->user->lang('Displays the percentage of customers retained from each period in the last 12 months.'));

        //Billing Cycles
        $billingcycles = array();

        include_once 'modules/billing/models/BillingCycleGateway.php';
        $gateway = new BillingCycleGateway();
        $iterator = $gateway->getBillingCycles(array(), array('order_value', 'ASC'));

        $monthlyBillingCycleId = false;

        while ($cycle = $iterator->fetch()) {
            if ($cycle->time_unit == 'm' && $cycle->amount_of_units == 1) {
                $monthlyBillingCycleId = $cycle->id;
                break;
            }
        }
        //Billing Cycles

        if ($monthlyBillingCycleId !== false) {
            // *** SQL to generate the the result set of the report ***

            $aCustomersRetained = array();

            //HOW MANY CUSTOMERS RETAINED FROM EACH PERIOD (MONTH AND YEAR). CONTINUES WITH BILLING ACTIVITY IN THE LATEST MONTH (HAS AN INVOICE CREATED, DUE OR PAID IN THE LAST 30 DAYS).
            $result = Capsule::table('users as u')
                ->selectRaw(
                    'DATE_FORMAT(u.dateActivated, "%M %Y") AS period, '
                    .'COUNT(*) as total'
                )
                ->where('u.groupid', 1)
                ->whereIn('u.id', function ($subQuery) use ($monthlyBillingCycleId) {
                    $subQuery->selectRaw('DISTINCT r.customerid')
                        ->from('recurringfee as r')
                        ->where('r.paymentterm', $monthlyBillingCycleId);
                })
                ->whereIn('u.id', function ($subQuery) {
                    $subQuery->selectRaw('DISTINCT i.customerid')
                        ->from('invoice as i')
                        ->whereRaw(
                            '(TO_DAYS(NOW()) - TO_DAYS(i.datecreated)) <= 30 '
                            .'OR (TO_DAYS(NOW()) - TO_DAYS(i.billdate)) <= 30 '
                            .'OR (TO_DAYS(NOW()) - TO_DAYS(i.datepaid)) <= 30'
                        );
                })
                ->groupBy('period')
                ->orderBy('u.dateActivated', 'ASC')
                ->get();

            foreach ($result as $row) {
                $aCustomersRetained[$row->period] = $row->total;
            }

            $aYears = array();

            //HOW MANY CUSTOMERS (GROUP 1), THAT CREATED THE ACCOUNT THAT PERIOD (MONTH AND YEAR), THAT HAD A MONTHLY RECURRING FEE (PAYMENT TERM 1) AND AT LEAST 1 INVOICE
            $result = Capsule::table('users as u')
                ->selectRaw(
                    'DATE_FORMAT(u.dateActivated, "%M %Y") AS period, '
                    .'COUNT(*) as total, '
                    .'YEAR(u.dateActivated) AS year, '
                    .'DATE_FORMAT(u.dateActivated, "%M") AS month'
                )
                ->where('u.groupid', 1)
                ->whereIn('u.id', function ($subQuery) use ($monthlyBillingCycleId) {
                    $subQuery->selectRaw('DISTINCT r.customerid')
                        ->from('recurringfee as r')
                        ->where('r.paymentterm', $monthlyBillingCycleId);
                })
                ->whereIn('u.id', function ($subQuery) {
                    $subQuery->selectRaw('DISTINCT i.customerid')
                        ->from('invoice as i');
                })
                ->groupBy('period')
                ->orderBy('u.dateActivated', 'ASC')
                ->get();

            foreach ($result as $row) {
                $mostRecentYear = $row->year;
                $aYears[$row->year][$row->month] = array(
                    'element' => $row->period,
                    'total'   => $row->total
                );
            }

            foreach ($aYears as $year => $aMonths) {
                $average = 0;
                $grouphidden = ($year != $mostRecentYear);
                $aGroup = array();

                foreach ($aMonths as $month => $values) {
                    if (!isset($aCustomersRetained[$values['element']])) {
                        $aCustomersRetained[$values['element']] = 0;
                    }
                    $average += round(($aCustomersRetained[$values['element']] * 100 / $values['total']), 2);
                    $aGroup[] = array($month, round(($aCustomersRetained[$values['element']] * 100 / $values['total']), 2)."%");
                }

                $aGroup[] = array('', '');
                $aGroup[] = array($this->user->lang('Average').' ('.$year.')', round(($average / (count($aGroup) - 1)), 2)."%");

                $groupid = "id-".$year;
                if (isset($aGroup)) {
                    $this->reportData[] = array(
                        "group"     => $aGroup,
                        "groupname" => $this->user->lang('New Customer Retention').' ('.$year.')',
                        "label"     => array($this->user->lang('Month'),$this->user->lang('Retention')),
                        "groupId"   => $groupid,
                        "isHidden"  => $grouphidden
                    );

                    unset($aGroup);
                }
            }

            //Display Report
            echo "<script type='text/javascript'>";
            echo "function ShowYears(obj){";
            echo "    var strYear = obj.value;";

            //Loop the number of years
            foreach ($aYears as $year => $aMonths) {
                echo "if (document.getElementById('id-$year') != null) {";
                echo "    if (strYear == '$year') {";
                echo "        document.getElementById('id-$year').style.display = '';";
                echo "    } else {";
                echo "        document.getElementById('id-$year').style.display = 'none';";
                echo "    }";
                echo "}";
            }

            echo "}";
            echo "</script>";
            echo $this->user->lang("Percentage of customers in each period (month and year) that created the account that period, had at least one monthly recurring fee and one invoice, and continues with billing activity in the latest month (has an invoice created, due or paid in the last 30 days).");
            echo "<br>";
            echo "<br>";
            echo "<br>";
            echo "<div style='margin-left:20px;'>";
            echo "    <form id='reportdropdown' method='GET'>";
            echo $this->user->lang('Select Year Range')."<br/>";
            echo "        <select id='passedyear' name='passedyear' class='normalSelect2 w-10' onChange='ShowYears(this);'>";
            
            //Loop the number of years
            foreach ($aYears as $year => $aMonths) {
                if ($year == $mostRecentYear) {
                    echo "<option value='".$year."' SELECTED>".$year."</option>";
                } else {
                    echo "<option value='".$year."'>".$year."</option>";
                }
            }

            echo "        </select>";
            echo "    </form>";
            echo "</div>";
        } else {
            echo $this->user->lang("You need to have a <b>Billing Cycle</b> defined with <b>Time Unit</b> in <b>Months</b> and <b>Amount of Units</b> equal to <b>1</b>.");
        }
    }
}
