<?php
use Illuminate\Database\Capsule\Manager as Capsule;

/**
 * Annual Revenue Report
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan D. Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 *
 *************************************************
 *   1.0 Initial Report Released.  - Juan D. Bolivar
 ************************************************
 */
require_once 'modules/billing/models/Currency.php';

/**
 * Annual_Revenue Report Class
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan D. Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 */
class Annual_Revenue extends Report
{

    private $lang;

    protected $featureSet = 'billing';
    public $hasgraph = true;

    function __construct($user = null, $customer = null)
    {
        $this->lang = lang('Annual Revenue');
        parent::__construct($user, $customer);
    }

    /**
     * Report Process Method
     *
     * @return null - direct output
     */
    function process()
    {
        //unset($this->session->annualrevenuedata);

        // Set the report information
        $this->SetDescription($this->user->lang('Displays revenue trends from previous month and year based on the date the transactions were made'));


        if (!isset($_REQUEST['passedyear'])) {
            $currentYear = date("Y");
        } else {
            $currentYear = $_REQUEST['passedyear'];
        }

        if (!isset($_REQUEST['graphtype'])) {
            $currentGraph = "revenue";
        } else {
            $currentGraph = $_REQUEST['graphtype'];
        }

        $thisYear = Date("Y");

        // Load the currency information
        $currency = new Currency($this->user);

        $currencyCode = ((isset($_REQUEST['currencycode']))? $_REQUEST['currencycode'] : $this->settings->get('Default Currency'));
        $currencyName = $currency->getName($currencyCode);

        $filter = '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'
            .'        '.$this->user->lang('Currency').': '
            .'        <select class="normalSelect2 w-25" name="currencycode" id="currencycode" value="'.CE_Lib::viewEscape($currencyCode).'" onChange="ChangeTable(document.getElementById(\'passedyear\').value, this.value);"> ';

        $isSelectedCurrencyInTheList = false;

        //Get all currencies of all invoices
        $result = Capsule::table('invoice as i')
            ->join('currency as c', 'c.abrv', '=', 'i.currency')
            ->distinct()
            ->select('c.abrv', 'c.name')
            ->orderBy('c.name', 'ASC')
            ->get();

        foreach ($result as $row) {
            if (!$isSelectedCurrencyInTheList && $currencyName < $row->name) {
                $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
                $isSelectedCurrencyInTheList = true;
            } elseif ($currencyCode == $row->abrv) {
                $isSelectedCurrencyInTheList = true;
            }

            $filter .= '<option value="'.$row->abrv.'" '.(($currencyCode == $row->abrv)? 'selected' : '').'>'.$row->name.'</option>';
        }

        if (!$isSelectedCurrencyInTheList) {
            $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
            $isSelectedCurrencyInTheList = true;
        }

        $filter .= '</select>';

        $graphdata = @$_GET['graphdata'];
        $getinvoices = @$_GET['getinvoices'];
        $firstTimeCurrency = true;
        $firstTimeYear = true;
        $firstTimeMonth = true;
        $monthCount = 0;
        $sumCount = 0;
        $monthAmountIn = 0;
        $sumAmountIn = 0;
        $monthAmountOut = 0;
        $sumAmountOut = 0;
        $monthAmount = 0;
        $sumAmount = 0;
        $lastMonthAmount = "na";
        $aYear = array();
        $aYearForGraph = array();
        $newCurrency = '';
        $newYear = 0;
        $newMonth = 0;
        $newTMonth = 0;
        $rawYears = array();
        $rawYear = array();
        $aYears = array();
        $aCurrencies = array();
        $minimumYear = $thisYear;
        $howManyYears = 0;

        $subquery = Capsule::table('invoicetransaction as it')
            ->select(
                'it.transactiondate',
                'it.invoiceid',
                'it.amount',
                'it.action'
            )
            ->distinct()
            ->where('it.accepted', 1)
            ->whereNotNull('it.transactiondate')
            ->where(function ($query) {
                $query->whereIn('it.action', ['charge', 'refund'])
                    ->orWhere(function ($subQuery) {
                        $subQuery->where('it.action', 'none')
                            ->where('it.response', 'like', '%of a variable payment%');
                    });
            });

        $result = Capsule::table(Capsule::raw("({$subquery->toSql()}) as it2"))
            ->mergeBindings($subquery) // very important to include bindings from subquery
            ->join('invoice as i', 'i.id', '=', 'it2.invoiceid')
            ->selectRaw(
                'DATE_FORMAT(it2.transactiondate, "%m") AS month, '
                .'YEAR(it2.transactiondate) AS year, '
                .'COUNT(*) AS counted, '
                .'SUM(ABS(it2.amount)) AS amount, '
                .'it2.action, '
                .'i.currency '
            )
            ->groupBy(
                'i.currency',
                Capsule::raw("DATE_FORMAT(it2.transactiondate, '%m')"),
                Capsule::raw("YEAR(it2.transactiondate)"),
                'it2.action'
            )
            ->orderBy('i.currency')
            ->orderBy('year')
            ->orderBy('month')
            ->orderBy('it2.action')
            ->get();

        foreach ($result as $row) {
            $ttMonth = $row->month;
            $tMonth = date("F", mktime(0, 0, 0, $ttMonth + 1, 0, 0));
            $tYear = $row->year;

            if ($tYear < $minimumYear) {
                $minimumYear = $tYear;
            }

            $tCurrency = $row->currency;

            if ($firstTimeCurrency) {
                $newCurrency = $tCurrency;
                $firstTimeCurrency = false;
            }

            if ($firstTimeYear) {
                $newYear = $tYear;
                $firstTimeYear = false;
            }

            if ($firstTimeMonth) {
                $newMonth = $ttMonth;
                $newTMonth = $tMonth;
                $firstTimeMonth = false;
            }


            if ($newCurrency != $tCurrency || $newYear != $tYear) {
                $monthAmountIn = $this->formatAmount($newCurrency, $monthAmountIn);
                $monthAmountOut = $this->formatAmount($newCurrency, $monthAmountOut);
                $formattedMonthAmount = $this->formatAmount($newCurrency, $monthAmount);

                $aYear[] = array(
                    "<a href=\"javascript:LoadInvoiceForMonthYear('".$newMonth."','".$newYear."','".$newCurrency."');\">".$newTMonth."</a>",
                    $monthCount,
                    $this->AverageInvoicePrice($monthCount, $monthAmount, $newCurrency),
                    $monthAmountIn,
                    $monthAmountOut,
                    $formattedMonthAmount,
                    $this->GetLastMonthDifference($lastMonthAmount, $newTMonth, $monthAmount, $newCurrency),
                    $this->GetLastYearDifference($rawYears[$newCurrency], $newTMonth, $newYear, $monthAmount, $newCurrency),
                );

                //same data as above but we are including the month as the key for graph
                $aYearForGraph[$newTMonth] = array(
                    $newTMonth,
                    $monthCount,
                    $this->AverageInvoicePrice($monthCount, $monthAmount, $newCurrency),
                    $monthAmountIn,
                    $monthAmountOut,
                    $formattedMonthAmount,
                    $this->GetLastMonthDifference($lastMonthAmount, $newTMonth, $monthAmount, $newCurrency),
                    $this->GetLastYearDifference($rawYears[$newCurrency], $newTMonth, $newYear, $monthAmount, $newCurrency),
                );

                $rawYear[$newTMonth] = $monthAmount;
                $lastMonthAmount = $monthAmount;

                $rawYears[$newCurrency][$newYear] = $rawYear;
                $aYears[$newCurrency][$newYear]["array"] = $aYear;
                $aYears[$newCurrency][$newYear]["arrayForGraph"] = $aYearForGraph;
                $aYears[$newCurrency][$newYear]["sumAmountIn"] = $sumAmountIn;
                $aYears[$newCurrency][$newYear]["sumAmountOut"] = $sumAmountOut;
                $aYears[$newCurrency][$newYear]["sumAmount"] = $sumAmount;
                $aYears[$newCurrency][$newYear]["sumCount"] = $sumCount;

                $newCurrency = $tCurrency;
                $newYear = $tYear;
                $newMonth = $ttMonth;
                $newTMonth = $tMonth;
                $rawYear = array();
                $aYear = array();
                $aYearForGraph = array();
                $monthCount = 0;
                $sumCount = 0;
                $monthAmountIn = 0;
                $sumAmountIn = 0;
                $monthAmountOut = 0;
                $sumAmountOut = 0;
                $monthAmount = 0;
                $sumAmount = 0;
            } elseif ($newMonth != $ttMonth) {
                $monthAmountIn = $this->formatAmount($newCurrency, $monthAmountIn);
                $monthAmountOut = $this->formatAmount($newCurrency, $monthAmountOut);
                $formattedMonthAmount = $this->formatAmount($newCurrency, $monthAmount);

                $aYear[] = array(
                    "<a href=\"javascript:LoadInvoiceForMonthYear('".$newMonth."','".$newYear."','".$newCurrency."');\">".$newTMonth."</a>",
                    $monthCount,
                    $this->AverageInvoicePrice($monthCount, $monthAmount, $newCurrency),
                    $monthAmountIn,
                    $monthAmountOut,
                    $formattedMonthAmount,
                    $this->GetLastMonthDifference($lastMonthAmount, $newTMonth, $monthAmount, $newCurrency),
                    $this->GetLastYearDifference($rawYears[$newCurrency], $newTMonth, $newYear, $monthAmount, $newCurrency),
                );

                //same data as above but we are including the month as the key for graph
                $aYearForGraph[$newTMonth] = array(
                    $newTMonth,
                    $monthCount,
                    $this->AverageInvoicePrice($monthCount, $monthAmount, $newCurrency),
                    $monthAmountIn,
                    $monthAmountOut,
                    $formattedMonthAmount,
                    $this->GetLastMonthDifference($lastMonthAmount, $newTMonth, $monthAmount, $newCurrency),
                    $this->GetLastYearDifference($rawYears[$newCurrency], $newTMonth, $newYear, $monthAmount, $newCurrency),
                );

                $rawYear[$newTMonth] = $monthAmount;
                $lastMonthAmount = $monthAmount;

                $newMonth = $ttMonth;
                $newTMonth = $tMonth;
                $monthCount = 0;
                $monthAmountIn = 0;
                $monthAmountOut = 0;
                $monthAmount = 0;
            }

            if ($row->action == 'refund') {
                $tAmount = - $row->amount;
                $monthAmountOut += $tAmount;
                $sumAmountOut += $tAmount;
            } else {
                $tAmount = $row->amount;
                $monthAmountIn += $tAmount;
                $sumAmountIn += $tAmount;
            }

            $monthAmount += $tAmount;
            $sumAmount += $tAmount;
            $tCounted = $row->counted;
            $monthCount+= $tCounted;
            $sumCount += $tCounted;
        }

        //Data for the last values calculated in the foreach
        $monthAmountIn = $this->formatAmount($newCurrency, $monthAmountIn);
        $monthAmountOut = $this->formatAmount($newCurrency, $monthAmountOut);
        $formattedMonthAmount = $this->formatAmount($newCurrency, $monthAmount);

        $aYear[] = array(
            "<a href=\"javascript:LoadInvoiceForMonthYear('".$newMonth."','".$newYear."','".$newCurrency."');\">".$newTMonth."</a>",
            $monthCount,
            $this->AverageInvoicePrice($monthCount, $monthAmount, $newCurrency),
            $monthAmountIn,
            $monthAmountOut,
            $formattedMonthAmount,
            $this->GetLastMonthDifference($lastMonthAmount, $newTMonth, $monthAmount, $newCurrency),
            $this->GetLastYearDifference($rawYears[$newCurrency], $newTMonth, $newYear, $monthAmount, $newCurrency),
        );

        //same data as above but we are including the month as the key for graph
        $aYearForGraph[$newTMonth] = array(
            $newTMonth,
            $monthCount,
            $this->AverageInvoicePrice($monthCount, $monthAmount, $newCurrency),
            $monthAmountIn,
            $monthAmountOut,
            $formattedMonthAmount,
            $this->GetLastMonthDifference($lastMonthAmount, $newTMonth, $monthAmount, $newCurrency),
            $this->GetLastYearDifference($rawYears[$newCurrency], $newTMonth, $newYear, $monthAmount, $newCurrency),
        );

        $rawYear[$newTMonth] = $monthAmount;
        $lastMonthAmount = $monthAmount;

        $rawYears[$newCurrency][$newYear] = $rawYear;
        $aYears[$newCurrency][$newYear]["array"] = $aYear;
        $aYears[$newCurrency][$newYear]["arrayForGraph"] = $aYearForGraph;
        $aYears[$newCurrency][$newYear]["sumAmountIn"] = $sumAmountIn;
        $aYears[$newCurrency][$newYear]["sumAmountOut"] = $sumAmountOut;
        $aYears[$newCurrency][$newYear]["sumAmount"] = $sumAmount;
        $aYears[$newCurrency][$newYear]["sumCount"] = $sumCount;

        if ($graphdata) {
            //this supports lazy loading and dynamic loading of graphs
            $this->reportData = $this->GraphData($currentGraph, $aYears[$currencyCode], $currencyCode);

            return;
        } elseif ($getinvoices) {
            $this->GetInvoiceForMonth($currencyCode);

            exit;
        }

        $yearCount = $thisYear - $minimumYear + 1;
        ;

        $aGroupIds = array();

        foreach ($aYears as $currencyKey => $aCurrency) {
            foreach ($aCurrency as $key => $aYear) {
                $grouphidden = true;
                $groupid = "id-" . $key . "-" . $currencyKey;

                if ($currencyCode == $currencyKey) {
                    if ($currentYear == $key && $currencyCode == $currencyKey) {
                        $grouphidden = false;
                    } else {
                        if (mb_substr($currentYear, 0, 4) == "Last") {
                            $tYears = mb_substr($currentYear, 4);

                            for ($y = 0; $y < $tYears; $y++) {
                                if ($key == date("Y") - $y) {
                                    $grouphidden = false;
                                }
                            }
                        }
                    }
                }

                $this->reportData[] = array(
                    "group"     => $aYear["array"],
                    "groupname" => $key.' ('.$currencyKey.')',
                    "label"     => array(
                        $this->user->lang('Month'),
                        $this->user->lang('Transactions'),
                        $this->user->lang('Avg. Price'),
                        $this->user->lang('Amount In'),
                        $this->user->lang('Amount Out'),
                        $this->user->lang('Month Total'),
                        "&Delta; " . $this->user->lang('Last Month'),
                        "&Delta; " . $this->user->lang('Last Year')
                    ),
                    "groupId"   => $groupid,
                    "isHidden"  => $grouphidden
                );

                $aGroupIds[] = $groupid;
                $formattedSumAmountIn = $this->formatAmount($currencyKey, $aYear["sumAmountIn"]);
                $formattedSumAmountOut = $this->formatAmount($currencyKey, $aYear["sumAmountOut"]);
                $formattedSumAmount = $this->formatAmount($currencyKey, $aYear["sumAmount"]);
                $aGroup = array();
                $aGroup[] = array($aYear["sumCount"], $formattedSumAmountIn, $formattedSumAmountOut, $formattedSumAmount, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
                $this->reportData[] = array(
                    "istotal"  => true,
                    "group"    => $aGroup,
                    "label"    => array(
                        $this->user->lang('Total Count'),
                        $this->user->lang('Total Amount In'),
                        $this->user->lang('Total Amount Out'),
                        $this->user->lang('Total Amount'),
                        ""
                    ),
                    "groupId"  => $groupid . "-totals",
                    "isHidden" => $grouphidden
                );

                $aGroupIds[] = $groupid . "-totals";

                unset($aGroup);
            }
        }

        //Display Report
        echo "\n\n<script type='text/javascript'>";

        echo "function ChangeTable(passedyear, currencycode){ ";

        foreach ($aGroupIds as $groupId) {
            echo "if (document.getElementById('$groupId') != null) {\n";
            echo "    document.getElementById('$groupId').style.display='none';\n";
            echo "}\n";
        }

        echo "    if (passedyear.substring(0,4)==\"Last\") {\n";
        echo "        yearsback = passedyear.substring(4);\n";
        echo "        for (x = 0; x < yearsback; x++) {\n";
        echo "            if (document.getElementById('id-'+(" . $thisYear . "-x)+'-'+currencycode) != null){ \n";
        echo "                document.getElementById('id-'+(" . $thisYear . "-x)+'-'+currencycode).style.display='';\n";
        echo "            };\n";
        echo "            if( document.getElementById('id-'+(" . $thisYear . "-x)+'-'+currencycode+'-totals') != null) {\n";
        echo "                document.getElementById('id-'+(" . $thisYear . "-x)+'-'+currencycode+'-totals').style.display='';\n";
        echo "            };\n";
        echo "        }\n";
        echo "    } else {\n";
        echo "        if (document.getElementById('id-'+passedyear+'-'+currencycode) != null) {\n";
        echo "            document.getElementById('id-'+passedyear+'-'+currencycode).style.display='';\n";
        echo "        };\n";
        echo "        if (document.getElementById('id-'+passedyear+'-'+currencycode+'-totals') != null) {\n";
        echo "            document.getElementById('id-'+passedyear+'-'+currencycode+'-totals').style.display='';\n";
        echo "        };\n";
        echo "    }\n";
        echo "    clientexec.populate_report('Annual_Revenue-Revenue','#myChartCanvas',{passedyear:passedyear, currencycode:currencycode});\n";
        echo "}\n";

        echo "function updateInvoicesForMonthYear(responseObj){\n";
        echo "    var responseArr = responseObj.responseText.split('||');\n";
        echo "    document.getElementById(responseArr[0]).style.display='';\n";
        echo "    document.getElementById(responseArr[0]).innerHTML=responseArr[1];\n";
        echo "}\n";

        echo "function LoadInvoiceForMonthYear(month,year,currencycode){\n";
        echo "    var url;\n";
        $url = "admin/index.php?noheaderfooter=1&getinvoices=1&month='+month+'&year='+year+'&currencycode='+currencycode+'&view=".CE_Lib::viewEscape($_REQUEST['view'])."&fuse=".CE_Lib::viewEscape($_REQUEST['fuse'])."&report=".urlencode($_REQUEST['report'])."&type=".CE_Lib::viewEscape($_REQUEST['type']);
        $url = mb_substr(CE_Lib::getSoftwareURL(), -1, 1) == "//" ? CE_Lib::getSoftwareURL().$url : CE_Lib::getSoftwareURL()."/".$url;
        echo "    url ='".$url."';\n";

        echo "    window.open(url, '', 'top=100, left=100, width=800, height=600, scrollbars=yes');\n";
        echo "}\n";


        echo "</script>";
        echo "\n\n";

        echo '<div style="margin-left:20px;"><b>'.$this->user->lang('Displays revenue trends from previous month and year based on the date the invoice was fully paid').'</b></div><br>';

        echo "<form id='reportdropdown' method='GET'>";
        echo "<input type='hidden' name='fuse' value='reports' />";
        echo "<input type='hidden' name='view' value='viewreport' />";
        echo "<input type='hidden' name='report' value='" . CE_Lib::viewEscape($_REQUEST['report']) . "' />";
        echo "<input type='hidden' name='type' value='" . CE_Lib::viewEscape($_REQUEST['type']) . "' />";

        echo "<div style='margin-left:20px;'>";
        echo "<b>" . $this->user->lang('Select Year Range') . "</b><br/>";
        echo "<select class='normalSelect2 w-25' id='passedyear' name='passedyear' onChange='ChangeTable(this.value, document.getElementById(\"currencycode\").value);'>";

        // If year count is 1, and we don't have any values for this year, we need to make year count 2 so it shows last year as we have values for that.
        if ($yearCount == 1 && !isset($aYears[$currencyCode][$thisYear])) {
            $yearCount = 2;
        }

        //Loop the number of years
        for ($x = 0; $x < $yearCount; $x++) {
            $xYear = $thisYear - $x;

            if ($currentYear == $xYear) {
                echo "<option value='" . $xYear . "' selected>" . $xYear . "</option>";
            } else {
                echo "<option value='" . $xYear . "'>" . $xYear . "</option>";
            }
        }

        //Create based on number of years of data available
        for ($x = 1; $x < $yearCount; $x++) {
            $value = $x + 1;

            if ($currentYear == "Last$value") {
                echo "<option value='Last$value' selected>Last $value Years&nbsp;&nbsp;&nbsp;</option>";
            } else {
                echo "<option value='Last$value'>Last $value Years&nbsp;&nbsp;&nbsp;</option>";
            }
        }

        echo "</select>";
        echo $filter;
        echo "</div>";
        echo "</form>";
    }

    /**
     * Function to generate the graph data
     *
     * @return null - direct output
     */
    function GraphData($graphType, $aYears, $currencyCode)
    {
        if (!isset($_REQUEST['passedyear'])) {
            $currentYear = date("Y");

            if (isset($_REQUEST['indashboard'])) {
                $currentYear = "Last2";
            }
        } else {
            $currentYear = $_REQUEST['passedyear'];
        }


        $yearCount = 0;
        $ydata = array();

        foreach ($aYears as $key => $aYear) {
            if ($currentYear == $key) {
                $ydata[] = array($key, $this->ReturnMonthsArray($aYear["arrayForGraph"], $graphType, $key, $currencyCode));
            } else {
                if (mb_substr($currentYear, 0, 4) == "Last") {
                    $tYears = mb_substr($currentYear, 4);

                    for ($y = 0; $y < $tYears; $y++) {
                        if ($key == date("Y") - $y) {
                            $ydata[] = array($key, $this->ReturnMonthsArray($aYear["arrayForGraph"], $graphType, $key, $currencyCode));
                        }
                    }
                }
            }
            $yearCount++;
        }

        //get currency symbol
        $currency = new Currency($this->user);

        //building graph data to pass back
        $graph_data = array(
            "xScale"  => "time",
            "yScale"  => "linear",
            "yType"   => "currency",
            "yPre"    => $currency->ShowCurrencySymbol($currencyCode),
            "yFormat" => "addcomma",
            "type"    => "line-dotted",
            "main"    => array()
        );

        foreach ($ydata as $y) {
            $year_data = array();
            $year_data['className'] = ".report_".$y[0];
            $year_data['data'] = array();

            foreach ($y[1] as $key => $monthtotal) {
                $month_data = array();

                if ($monthtotal == "") {
                    $monthtotal = "0";
                }

                $month_data["x"] = $key;
                $month_data["y"] = sprintf("%01.".$currency->getPrecision($currencyCode)."f", round((float)$monthtotal, $currency->getPrecision($currencyCode)));
                $year_data['data'][] = $month_data;
            }

            $graph_data["main"][] = $year_data;
        }

        return json_encode($graph_data);
    }

    //*********************************************
    // Custom Function Definitions for this report
    //*********************************************
    function AverageInvoicePrice($invoices, $price, $currencyCode)
    {
        $invoices = (float) $invoices;

        if ($invoices > 0) {
            $average = ((float) $price) / $invoices;
        } else {
            $average = (float) 0;
        }

        return $this->formatAmount($currencyCode, $average);
    }

    function GetLastMonthDifference($lastmonthamount, $month, $amount, $currencyCode)
    {
        if ($lastmonthamount == "na") {
            return "<b>na</b>";
        } else {
            $retValue = (float) $amount - (float) $lastmonthamount;
            return $this->formatAmount($currencyCode, $retValue, true);
        }
    }

    function GetLastYearDifference($rawYears, $month, $year, $amount, $currencyCode)
    {
        if (isset($rawYears[$year - 1][$month])) {
            $lastYearPrice = $rawYears[$year - 1][$month];
            $retValue = (float) $amount - (float) $lastYearPrice;
            return $this->formatAmount($currencyCode, $retValue, true);
        } else {
            return "<b>na</b>";
        }
    }

    function formatAmount($currencyCode, $amount, $isDelta = false, $html = false)
    {
        $currency = new Currency($this->user);
        $amountFormatted = $currency->format($currencyCode, $amount, true, "NONE", $html);

        if ($amount < 0) {
            $amountFormatted = "<font color='red'>" . $amountFormatted . "</font>";

            if ($isDelta) {
                $amountFormatted = '(' . $amountFormatted . ')';
            }
        }

        return $amountFormatted;
    }

    function CleanFloat($strFormattedAmount, $currencyCode)
    {
        //remove formatting
        $currency = new Currency($this->user);
        $strFormattedAmount = str_replace(
            array(
                $currency->ShowCurrencySymbol($currencyCode, false),
                $currency->getThousandsSeparator($currencyCode),
                $currency->getDecimalsSeparator($currencyCode),
            ),
            array(
                "",
                "",
                ".",
            ),
            $strFormattedAmount
        );

        $strFormattedAmount = strip_tags($strFormattedAmount);
        $strFormattedAmount = str_replace(array("(", ")"), array('', ''), $strFormattedAmount);
        $strFormattedAmount = trim($strFormattedAmount);

        return (float) $strFormattedAmount;
    }

    function ReturnMonthsArray($aMonth, $graphType, $currentYear, $currencyCode)
    {
        //Get all months values
        //Loop through 12 months using long date description and determine
        //if aMonth contains that month.  If not return 0.0

        $aReturn = array();

        for ($x = 1; $x <= 12; $x++) {
            $tstrMonth = date("F", mktime(0, 0, 0, $x + 1, 0, 0));
            $date = date("Y-m-d", mktime(0, 0, 0, $x, 1, $currentYear));
            $valIndex = 5;

            if (!isset($aMonth[$tstrMonth])) {
                //what should we be doing here?
                $aReturn[$date] = "";
            } elseif ($this->CleanFloat($aMonth[$tstrMonth][$valIndex], $currencyCode) == "na") {
                $aReturn[$date] = "";
            } else {
                $tempY = $this->CleanFloat($aMonth[$tstrMonth][$valIndex], $currencyCode);
                $aReturn[$date] = $tempY;
            }

            if (($currentYear == date("Y")) && (date("F") == $tstrMonth)) {
                return $aReturn;
            }
        }

        return $aReturn;
    }

    //Functions called asynchrounsly
    //Most of this comes from the old Income Graph first developed by Kevin Grubbs
    //Return Formatted HMTL
    function GetInvoiceForMonth($currencyCode)
    {

        // date format of the paid date
        // uses the PHP date() format syntax
        $strDateFormat = "j M Y";

        include_once 'modules/billing/models/Currency.php';
        $currency = new Currency($this->user);

        $nCurrentYear = $_GET["year"];
        $nCurrentMonth = $_GET["month"];

        //Display Report
        $tMonth = date("F", mktime(0, 0, 0, $nCurrentMonth + 1, 0, 0));

        echo "<H1>" . $tMonth . " " . CE_Lib::viewEscape($nCurrentYear) . "</h1></br>";

        echo "<style>
            table {
                width:100%;
            }
            table, th, td {
                border: 1px solid black;
                border-collapse: collapse;
            }
            th, td {
                padding: 5px;
                text-align: left;
            }
            table#t01 tr:nth-child(even) {
                background-color: #eee;
            }
            table#t01 tr:nth-child(odd) {
               background-color:#fff;
            }
            table#t01 th	{
                background-color: white;
                color: black;
            }
            </style>";

        echo "<table id='t01'>";
        echo "<tr><th>". $this->user->lang('Invoice') . "</th><th>" . $this->user->lang('Customer') . "</th><th>" . $this->user->lang('Amount') . "</th><th>" . $this->user->lang('Type') . "</th><th>" . $this->user->lang('Transaction Date') . "</th></tr>";

        //initialize
        $amountIn = 0;
        $amountOut = 0;
        $revenueTotal = 0;
        $monthTransactions = 0;

        //SQL to generate the result set of the report.
        $result = Capsule::table('invoicetransaction as it')
            ->join('invoice as i', 'i.id', '=', 'it.invoiceid')
            ->selectRaw(
                'DISTINCT i.id, '
                .'i.customerid, '
                .'ABS(it.amount) AS amount, '
                .'it.action, '
                .'it.transactiondate'
            )
            ->where('it.accepted', 1)
            ->whereNotNull('it.transactiondate')
            ->where(function ($query) {
                $query->whereIn('it.action', ['charge', 'refund'])
                    ->orWhere(function ($subQuery) {
                        $subQuery->where('it.action', 'none')
                            ->where('it.response', 'like', '%of a variable payment%');
                    });
            })
            ->whereYear('it.transactiondate', $nCurrentYear)
            ->whereMonth('it.transactiondate', $nCurrentMonth)
            ->where('i.currency', $currencyCode)
            ->orderBy('it.transactiondate')
            ->orderBy('it.action')
            ->get();

        foreach ($result as $row) {
            $thisMonth = date("n", strtotime($row->transactiondate));

            // Get list of use name
            $title = '';

            if ($row->customerid != 0) {
                $tUser = new User($row->customerid);

                if ($tUser->IsOrganization()) {
                    $lastName = $tUser->getOrganization();

                    if (strlen($lastName) > 20) {
                        $dCustomerName = mb_substr($lastName, 0, 20) . "...";
                        $title = "title=\"$lastName\"";
                    } else {
                        $dCustomerName = $lastName;
                    }
                } else {
                    $dCustomerName = $tUser->getFirstName() . " " . $tUser->getLastName();
                    $title = "";
                }

                // build a link to the customer profile
                $CustomerLink = "<a href=\"index.php?fuse=clients&controller=userprofile&frmClientID=" . $row->customerid . "&view=profileinvoices\" $title target=blank>" . CE_Lib::viewEscape($dCustomerName) . "</a>";

                // build the Print to PDF Link
                $printInvoiceLink = "<a href=index.php?fuse=billing&frmClientID=" . $row->customerid . "&controller=invoice&view=invoice&profile=1&amp;invoiceid=" . $row->id . " target=blank>" . $row->id . "</a>";
            } else {
                $dCustomerName = $this->user->lang('DELETED');
                $CustomerLink = CE_Lib::viewEscape($dCustomerName);
                $printInvoiceLink = $row->id;
            }

            if ($row->action == 'refund') {
                $tAmount = - $row->amount;
                $amountOut += $tAmount;
                $type = 'Refund';
            } else {
                $tAmount = $row->amount;
                $amountIn += $tAmount;
                $type = 'Payment';
            }

            $strAmountFormatted = $this->formatAmount($currencyCode, $tAmount, false, true);
            $revenueTotal += $tAmount;
            $aGroup = array();
            $aGroup[] = array($printInvoiceLink, $CustomerLink, $strAmountFormatted, date($strDateFormat, strtotime($row->transactiondate)));
            $monthTransactions++;
            $strAmountInFormatted = $this->formatAmount($currencyCode, $amountIn, false, true);
            $strAmountOutFormatted = $this->formatAmount($currencyCode, $amountOut, false, true);
            $strRevenueTotalFormatted = $this->formatAmount($currencyCode, $revenueTotal, false, true);

            echo "<tr><td>" . $printInvoiceLink . "</td><td>" . $CustomerLink . "</td><td>" . $strAmountFormatted . "</td><td>" . $type . "</td><td>" . date($strDateFormat, strtotime($row->transactiondate)) . "</td></tr>";
        }

        echo "</table>";

        if ($monthTransactions > 0) {
            echo "</br><table>";
            echo "<tr><th>" . $this->user->lang('Transactions') . "</th><th>" . $this->user->lang('Amount In') . "</th><th>" . $this->user->lang('Amount Out') . "</th><th>" . $this->user->lang('Month Total') . "</th></tr>";
            echo "<tr><td>" . $monthTransactions . "</td><td>" . $strAmountInFormatted . "</td><td>" . $strAmountOutFormatted . "</td><td>" . $strRevenueTotalFormatted . "</td></tr>";
            echo "</table>";
        }
    }
}
