<?php
use Illuminate\Database\Capsule\Manager as Capsule;

/**
 * Monthly Revenue Report
 *
 * @category Report
 * @package  ClientExec
 * @author   Jason Yates <jason@clientexec.com>
 * @license  ClientExec License
 * @version  1.5
 * @link     http://www.clientexec.com
 *
 *************************************************
 *   1.0 Initial Report Released.  - Bart Wegrzyn
 *   1.1 Fixed to account for yearly, semi-yearly,  quarterly, and monthly pricing.  - Shane Sammons
 *   1.2 Only show those packages that have recurring prices not all packages.  - Alberto Vasquez
 *   1.3 Only show those packages for active customers.  - Alberto Vasquez
 *   1.4 Refactored to show revenue from addons as well - Alejandro Pedraza
 *   1.5 Refactored report to adhere to Pear standards & Updated the package names that are shown - Jason Yates
 ************************************************
 */

require_once 'modules/billing/models/Currency.php';
require_once 'modules/billing/models/BillingType.php';
require_once('modules/clients/models/DomainNameGateway.php');

/**
 * Monthly_Revenue Report Class
 *
 * @category Report
 * @package  ClientExec
 * @author   Jason Yates <jason@clientexec.com>
 * @license  ClientExec License
 * @version  1.5
 * @link     http://www.clientexec.com
 */
class Monthly_Revenue extends Report
{
    private $lang;

    protected $featureSet = 'billing';

    function __construct($user = null, $customer = null)
    {
        $this->lang = lang('Monthly Revenue');
        parent::__construct($user, $customer);
    }

    /**
     * Report Process Method
     *
     * @return null - direct output
     */
    function process()
    {
        //Billing Cycles
        $billingcycles = array();

        include_once 'modules/billing/models/BillingCycleGateway.php';
        $gateway = new BillingCycleGateway();
        $iterator = $gateway->getBillingCycles(array(), array('order_value', 'ASC'));

        while ($cycle = $iterator->fetch()) {
            $billingcycles[$cycle->id] = array(
                'name'            => $this->user->lang($cycle->name),
                'time_unit'       => $cycle->time_unit,
                'amount_of_units' => $cycle->amount_of_units
            );
        }
        //Billing Cycles

        include_once 'modules/admin/models/StatusAliasGateway.php' ;

        // Set the report information
        $this->SetDescription($this->user->lang('Displays total recurring transactions broken down by packages with the sum and expected monthly revenue from each.'));

        // Load the currency information
        $currency = new Currency($this->user);

        $currencyCode = ((isset($_REQUEST['currencycode']))? $_REQUEST['currencycode'] : $this->settings->get('Default Currency'));
        $currencyName = $currency->getName($currencyCode);

        $filter = '<form id="report" method="GET">'
            .'    <div style="text-align:center">'
            .'        '.$this->user->lang('Currency').': '
            .'        <select class="normalSelect2 w-20" name="currencycode" id="currencycode" value="'.CE_Lib::viewEscape($currencyCode).'" onChange="ChangeTable(this.value);"> ';

        //Get all currencies of all users
        $result = Capsule::table('users as u')
            ->join('currency as c', 'c.abrv', '=', 'u.currency')
            ->select('c.abrv', 'c.name')
            ->distinct()
            ->orderBy('c.name', 'ASC')
            ->get();

        $isSelectedCurrencyInTheList = false;

        foreach ($result as $row) {
            if (!$isSelectedCurrencyInTheList && $currencyName < $row->name) {
                $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
                $isSelectedCurrencyInTheList = true;
            } elseif ($currencyCode == $row->abrv) {
                $isSelectedCurrencyInTheList = true;
            }
            $filter .= '<option value="'.$row->abrv.'" '.(($currencyCode == $row->abrv)? 'selected' : '').'>'.$row->name.'</option>';
        }
        if (!$isSelectedCurrencyInTheList) {
            $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
            $isSelectedCurrencyInTheList = true;
        }

        $filter .= '</select>';
        $filter .= '    </div>'
            .'</form>'
            .'</br>'
            .'<script type="text/javascript">'
            .'    function ChangeTable(currencycode){'
            .'        location.href="index.php?fuse=reports&view=viewreport&controller=index&report=Monthly+Revenue&type=Revenue&currencycode="+currencycode;'
            .'    }'
            .'</script>';
        echo $filter;

        $userStatuses = StatusAliasGateway::userActiveAliases($this->user);
        $packageStatuses = StatusAliasGateway::packageActiveAliases($this->user);

        //SQL to generate the the result set of the report
        $result = Capsule::table('domains as d')
            ->leftJoin('recurringfee as rf', function ($join) {
                $join->on('rf.appliestoid', '=', 'd.id')
                     ->where('rf.billingtypeid', '=', -1);
            })
            ->join('users as u', 'd.customerid', '=', 'u.id')
            ->selectRaw(
                'COUNT(*) AS counted, d.Plan, rf.paymentterm, d.use_custom_price, d.custom_price'
            )
            ->where('rf.paymentterm', '<>', 0)
            ->whereRaw('IFNULL(rf.recurring, 0) <> 0')
            ->whereIn('u.status', $userStatuses)
            ->whereIn('d.status', $packageStatuses)
            ->where('u.currency', $currencyCode)
            ->groupBy('rf.paymentterm', 'd.Plan', 'd.use_custom_price', 'd.custom_price')
            ->get();

        $expectedrevenueTotal = 0;
        $sumpertotalpackagesTotal = 0;

        include_once 'modules/billing/models/Prices.php';
        $prices = new Prices();

        // Just in case we have domains
        $dng = new DomainNameGateway($this->user);

        foreach ($result as $row) {
            $result2 = Capsule::table('package as p')
                ->join('promotion as g', 'p.planid', '=', 'g.id')
                ->select('p.planname as planname', 'g.name as groupname', 'p.pricing', 'g.type')
                ->where('p.id', $row->Plan)
                ->get();

            foreach ($result2 as $row2) {
                $append = ' ('.$billingcycles[$row->paymentterm]['name'].')';

                $row2->pricing = $prices->getPricing(PRODUCT_PRICE, $row->Plan, $currencyCode, $row2->pricing);
                $pricing = unserialize($row2->pricing);

                $packagePrice = 0;

                if (is_array($pricing)) {
                    if ($row2->type == 3) {  // Domain Type
                        $pricingInformation = array();

                        foreach ($pricing as $key => $value) {
                            $pricingInformation[$key] = $value;
                        }

                        $pricingArray = array_pop($pricingInformation['pricedata']);

                        if ($row->paymentterm != 0 && $billingcycles[$row->paymentterm]['time_unit'] == 'y' && $billingcycles[$row->paymentterm]['amount_of_units'] != 0 && isset($pricingArray[$row->paymentterm]['price'])) {
                            $packagePrice = ($pricingArray[$row->paymentterm]['price']/$billingcycles[$row->paymentterm]['amount_of_units'])/12;
                        }
                    } else {
                        if ($row->paymentterm != 0 && $billingcycles[$row->paymentterm]['amount_of_units'] != 0 && isset($pricing['price'.$row->paymentterm]) && is_numeric($pricing['price'.$row->paymentterm])) {
                            switch ($billingcycles[$row->paymentterm]['time_unit']) {
                                case 'd':
                                    $packagePrice = ($pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'])*30;
                                    break;
                                case 'w':
                                    $packagePrice = (($pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'])/7)*30;
                                    break;
                                case 'm':
                                    $packagePrice = $pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'];
                                    break;
                                case 'y':
                                    $packagePrice = ($pricing['price'.$row->paymentterm]/$billingcycles[$row->paymentterm]['amount_of_units'])/12;
                                    break;
                            }
                        }
                    }
                }

                if ($row->paymentterm != 0 && $row->use_custom_price && $billingcycles[$row->paymentterm]['amount_of_units'] != 0) {
                    switch ($billingcycles[$row->paymentterm]['time_unit']) {
                        case 'd':
                            $packagePrice = ($row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'])*30;
                            break;
                        case 'w':
                            $packagePrice = (($row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'])/7)*30;
                            break;
                        case 'm':
                            $packagePrice = $row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'];
                            break;
                        case 'y':
                            $packagePrice = ($row->custom_price/$billingcycles[$row->paymentterm]['amount_of_units'])/12;
                            break;
                    }

                    $append .= " (overridden)";
                }

                $tMonthlyRevenue = $packagePrice * $row->counted;
                //echo "Plan Name: ".
                $tPackageName = $row2->planname." / ".$row2->groupname.$append;
                //echo "Quantity: ".
                $tPackageCount = $row->counted;

                $tExpectedRevenue = $currency->format($currencyCode, $tMonthlyRevenue, true);
                $expectedrevenueTotal += $tMonthlyRevenue;
                $sumpertotalpackagesTotal += $tPackageCount;
                $aGroup[] = array($tPackageName,$tPackageCount,$tExpectedRevenue);

                // NOTE: remember the addon cycle can be different than the package's
                $result3 = Capsule::table('recurringfee as rf')
                    ->leftJoin('users as u', 'u.id', '=', 'rf.customerid')
                    ->leftJoin('domains as d', function ($join) use ($packageStatuses) {
                        $join->on('rf.appliestoid', '=', 'd.id')
                             ->leftJoin('recurringfee as rrff', function ($subJoin) {
                                 $subJoin->on('rrff.appliestoid', '=', 'd.id')
                                         ->where('rrff.billingtypeid', -1);
                             });
                    })
                    ->selectRaw('SUM(rf.amount * rf.quantity) AS total, rf.paymentterm')
                    ->where('rf.billingtypeid', BILLINGTYPE_PACKAGE_ADDON)
                    ->where('d.Plan', $row->Plan)
                    ->whereIn('d.status', $packageStatuses)
                    ->where('rrff.paymentterm', $row->paymentterm)
                    ->where('u.currency', $currencyCode)
                    ->groupBy('rf.paymentterm')
                    ->get();

                foreach ($result3 as $row3) {
                    if ($row3->paymentterm != 0 && $billingcycles[$row3->paymentterm]['amount_of_units'] != 0) {
                        switch ($billingcycles[$row3->paymentterm]['time_unit']) {
                            case 'd':
                                $row3->total = ($row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'])*30;
                                break;
                            case 'w':
                                $row3->total = (($row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'])/7)*30;
                                break;
                            case 'm':
                                $row3->total = $row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'];
                                break;
                            case 'y':
                                $row3->total = ($row3->total/$billingcycles[$row3->paymentterm]['amount_of_units'])/12;
                                break;
                        }
                    } else {
                        continue;
                    }

                    $append = ' ('.$billingcycles[$row3->paymentterm]['name'].')';
                    $tExpectedRevenue = $currency->format($currencyCode, $row3->total, true);
                    $expectedrevenueTotal += $row3->total;
                    $aGroup[] = array("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; Add-ons $append", '', $tExpectedRevenue);
                }
            }
        }

        if (isset($aGroup)) {
            $aGroup[] = array("","","");

            $expectedrevenueTotal = $currency->format($currencyCode, $expectedrevenueTotal, true);
            $aGroup[] = array("<b>".$this->user->lang("Totals")."</b>","<b>".$sumpertotalpackagesTotal."</b>", "<b>".$expectedrevenueTotal."</b>");

            $this->reportData[] = array(
              "group" => $aGroup,
              "groupname" => $this->user->lang("Package Names"),
              "label" => array($this->user->lang('Package Name'),$this->user->lang('Total Packages'),$this->user->lang('Expected Monthly Revenue')),
              "groupId" => "",
              "isHidden" => false);
            unset($aGroup);
        }
    }
}
