<?php
use Illuminate\Database\Capsule\Manager as Capsule;

/**
 * State Total Revenue Report
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 */

require_once 'modules/admin/models/Package.php';
require_once 'modules/clients/models/UserPackage.php';
require_once 'modules/billing/models/Currency.php';
require_once 'modules/clients/models/User.php';
require_once 'modules/billing/models/Invoice.php';

/**
 * Revenue_By_State Report Class
 *
 * @category Report
 * @package  ClientExec
 * @author   Juan Bolivar <juan@clientexec.com>
 * @license  ClientExec License
 * @version  1.0
 * @link     http://www.clientexec.com
 */
class Revenue_By_State extends Report
{
    private $lang;

    protected $featureSet = 'billing';

    function __construct($user = null, $customer = null)
    {
        $this->lang = lang('Revenue By State');
        parent::__construct($user, $customer);
    }

    /**
     * Report Process Method
     *
     * @return null - direct output
     */
    function process()
    {
        @set_time_limit(0);

        // Set the report information
        $this->SetDescription($this->user->lang('Displays the total revenue between 2 dates broken by state of the clients of the selected country.'));

        // Load the country information
        $countries = new Countries($this->user);
        $countryCode = ((isset($_REQUEST['countrycode']))? $_REQUEST['countrycode'] : $this->settings->get('Default Country'));
        $countryName = $countries->validCountryCode($countryCode, false, 'name');

        if (isset($_REQUEST['startdate'])) {
            $startDateArray = explode('/', $_REQUEST['startdate']);

            if ($this->settings->get('Date Format') == 'm/d/Y') {
                $tempStartDate = date("Y-m-d", mktime(0, 0, 0, $startDateArray[0], $startDateArray[1], $startDateArray[2]));
            } else {
                $tempStartDate = date("Y-m-d", mktime(0, 0, 0, $startDateArray[1], $startDateArray[0], $startDateArray[2]));
            }
        } else {
            $tempStartDate = date("Y-m-d", mktime(0, 0, 0, date("m"), 1, date("Y")-1));
        }

        if (isset($_REQUEST['enddate'])) {
            $endDateArray = explode('/', $_REQUEST['enddate']);

            if ($this->settings->get('Date Format') == 'm/d/Y') {
                $tempEndDate = date("Y-m-d", mktime(0, 0, 0, $endDateArray[0], $endDateArray[1], $endDateArray[2]));
            } else {
                $tempEndDate = date("Y-m-d", mktime(0, 0, 0, $endDateArray[1], $endDateArray[0], $endDateArray[2]));
            }
        } else {
            $tempEndDate = date("Y-m-d");
        }

        // Load the currency information
        $currency = new Currency($this->user);
        $currencyCode = ((isset($_REQUEST['currencycode']))? $_REQUEST['currencycode'] : $this->settings->get('Default Currency'));
        $currencyName = $currency->getName($currencyCode);

        // Get id of the country custom field
        $result = Capsule::table('customuserfields')
            ->where('type', typeCOUNTRY)
            ->value('id');

        $countryCustomFieldId = $result;

        // Get id of the state custom field
        $result = Capsule::table('customuserfields')
            ->where('type', typeSTATE)
            ->value('id');

        $stateCustomFieldId = $result;

        //Get all paid invoices amount from clients of the selected country paid the selected date range with the selected currency
        $result = Capsule::table('invoice as i')
            ->join('users as u', 'u.id', '=', 'i.customerid')
            ->join('user_customuserfields as ucuf', 'ucuf.userid', '=', 'u.id')
            ->join('currency as c', 'c.abrv', '=', 'i.currency')
            ->leftJoin('user_customuserfields as ucuf2', 'ucuf2.userid', '=', 'u.id')
            ->selectRaw(
                'ucuf2.value,
                 SUM(i.amount) AS totalAmount,
                 SUM(i.amount - i.subtotal) AS totalTax,
                 COUNT(*) AS totalNumber'
            )
            ->where('i.status', INVOICE_STATUS_PAID)
            ->where('ucuf.customid', $countryCustomFieldId)
            ->where('ucuf.value', $countryCode)
            ->where('ucuf2.customid', $stateCustomFieldId)
            ->where('i.datepaid', '>=', $tempStartDate)
            ->where('i.datepaid', '<=', $tempEndDate . ' 23:59:59')
            ->where('i.currency', $currencyCode)
            ->groupBy('ucuf2.value')
            ->orderByDesc('totalAmount')
            ->get();

        $subGroup = array();
        $totalAmount = 0;
        $totalTax = 0;
        $totalInvoices = 0;

        foreach ($result as $row) {
            $totalAmount += $row->totalAmount;
            $totalTax += $row->totalTax;
            $totalInvoices += $row->totalNumber;

            $states = new States($this);
            $stateProvince = $states->validStateCode($countryCode, $row->value, 'iso', false, 'name');
            $formattedAmount = $currency->format($currencyCode, $row->totalAmount, true, false);
            $formattedTax = $currency->format($currencyCode, $row->totalTax, true, false);

            $subGroup[] = array(
                (isset($stateProvince))? $stateProvince : '---',
                $formattedAmount,
                $formattedTax,
                $row->totalNumber
            );
        }

        $subGroup[] = array('', '', '', '');
        $formattedTotalAmount = $currency->format($currencyCode, $totalAmount, true, false);
        $formattedTotalTax = $currency->format($currencyCode, $totalTax, true, false);
        $subGroup[] = array('<b>'.$this->user->lang('Totals').'</b>', '<b>'.$formattedTotalAmount.'</b>', '<b>'.$formattedTotalTax.'</b>', '<b>'.$totalInvoices.'</b>');

        $labels = array(
            $this->user->lang('State/Province'),
            $this->user->lang('Total Paid Amount'),
            $this->user->lang('Total Paid Tax'),
            $this->user->lang('Number of Paid Invoices')
        );

        if (isset($_REQUEST['download']) && $_REQUEST['download'] == 1) {
            $this->download($labels, $subGroup, $countryName.' '.'total revenue from '.$tempStartDate.' to '.$tempEndDate.' for '.$currencyCode.'.csv');
        }

        $filter = '<form id="report" method="GET">'
            .'    <div style="text-align:center">'
            .'        '.$this->user->lang('Country').': '
            .'        <select class="normalSelect2 w-20" name="countrycode" id="countrycode" value="'.CE_Lib::viewEscape($countryCode).'" > ';

        //Get all countries with paid invoices
        $result = Capsule::table('country as c')
            ->join('user_customuserfields as ucuf', function ($join) use ($countryCustomFieldId) {
                $join->on('ucuf.value', '=', 'c.iso')
                     ->where('ucuf.customid', '=', $countryCustomFieldId);
            })
            ->join('invoice as i', function ($join) {
                $join->on('i.customerid', '=', 'ucuf.userid')
                     ->where('i.status', '=', INVOICE_STATUS_PAID);
            })
            ->select('c.name', 'c.iso')
            ->distinct()
            ->orderBy('c.name', 'asc')
            ->get();

        $isSelectedCountryInTheList = false;

        foreach ($result as $row) {
            if (!$isSelectedCountryInTheList && $countryName < $row->name) {
                $filter .= '<option value="'.$countryCode.'" selected>'.$countryName.'</option>';
                $isSelectedCountryInTheList = true;
            } elseif ($countryCode == $row->iso) {
                $isSelectedCountryInTheList = true;
            }
            $filter .= '<option value="'.$row->iso.'" '.(($countryCode == $row->iso)? 'selected' : '').'>'.$row->name.'</option>';
        }
        if (!$isSelectedCountryInTheList) {
            $filter .= '<option value="'.$countryCode.'" selected>'.$countryName.'</option>';
            $isSelectedCountryInTheList = true;
        }

        $filter .= '</select>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;';

        $filter .= '        '.$this->user->lang('Date Paid').': '
            .'        <input class="datepicker" type="text" name="startdate" id="startdate" value=\''.CE_Lib::db_to_form($tempStartDate, $this->settings->get('Date Format'), "/").'\' />'
            .' - '
            .'<input class="datepicker" type="text" name="enddate" id="enddate" value=\''.CE_Lib::db_to_form($tempEndDate, $this->settings->get('Date Format'), "/").'\' />&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;';

        $filter .= '        '.$this->user->lang('Currency').': '
            .'        <select class="normalSelect2 w-20" name="currencycode" id="currencycode" value="'.CE_Lib::viewEscape($currencyCode).'" > ';

        //Get all currencies when were paid invoices from clients of the selected country
        $result = Capsule::table('invoice as i')
            ->join('users as u', 'u.id', '=', 'i.customerid')
            ->join('user_customuserfields as ucuf', 'ucuf.userid', '=', 'u.id')
            ->join('currency as c', 'c.abrv', '=', 'i.currency')
            ->where('i.status', INVOICE_STATUS_PAID)
            ->where('ucuf.customid', $countryCustomFieldId)
            ->where('ucuf.value', $countryCode)
            ->select('c.abrv', 'c.name')
            ->distinct()
            ->orderBy('c.name', 'asc')
            ->get();

        $isSelectedCurrencyInTheList = false;

        foreach ($result as $row) {
            if (!$isSelectedCurrencyInTheList && $currencyName < $row->name) {
                $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
                $isSelectedCurrencyInTheList = true;
            } elseif ($currencyCode == $row->abrv) {
                $isSelectedCurrencyInTheList = true;
            }
            $filter .= '<option value="'.$row->abrv.'" '.(($currencyCode == $row->abrv)? 'selected' : '').'>'.$row->name.'</option>';
        }
        if (!$isSelectedCurrencyInTheList) {
            $filter .= '<option value="'.$currencyCode.'" selected>'.$currencyName.'</option>';
            $isSelectedCurrencyInTheList = true;
        }

        $filter .= '</select>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;';

        $filter .= '        <input type=button name=search class="cm-btns orange" value=\''.$this->user->lang('Search').'\' onclick="ChangeTable(document.getElementById(\'countrycode\').value, document.getElementById(\'startdate\').value, document.getElementById(\'enddate\').value, document.getElementById(\'currencycode\').value, 0);">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;';

        $filter .= '        <button class="cm-btns orange" type="button" data-loading-text="Loading..." onclick="ChangeTable(document.getElementById(\'countrycode\').value, document.getElementById(\'startdate\').value, document.getElementById(\'enddate\').value, document.getElementById(\'currencycode\').value, 1);">'.$this->user->lang("Download .csv").'</button>';

        $filter .= '    </div>'
            .'</form>'
            .'</br>'
            .'<script type="text/javascript">'
            .'    function ChangeTable(countrycode, startdate, enddate, currencycode, download){'
            .'        location.href="index.php?fuse=reports&view=viewreport&controller=index&report=Revenue+By+State&type=Revenue&countrycode="+countrycode+"&startdate="+encodeURIComponent(startdate)+"&enddate="+encodeURIComponent(enddate)+"&currencycode="+currencycode+"&download="+download;'
            .'    }'
            .'</script>';
        echo $filter;

        $this->reportData[] = array(
            "group" => $subGroup,
            "groupname" => "",
            "label" => $labels
        );
    }
}
